/*

	filelist.c      IBM Internal Use Only

 

	This file contains two C functions:

 

	        int     findfirst( char* filespec, char* filename );

	        int     findnext( char* filename );

 

	findfirst is given a filespec which may contain wildcard

	characters.  It searches the directory specified by filespec

	for files which match.  If at least one such file is found,

	the first matching file is returned.  (Files are ordered

	alphabetically.)  Matching entries which are directories are

	not returned.

 

	findnext returns the next matching file found.

 

	Both functions return -1 if there are no matches (or no

	remaining matches in the case of findnext) or 0 if a match

	was returned.

 

	If the preprocessor variable FULL_LIST is defined when this

	program is compiled, it will return all directory entries

	including those which are themselves directories.  Otherwise,

	it will not return any matching entries which are directories.

 

	Created on 8 March 1991 by A.D. Drumm

 

	Modified:

 

	27 March 1991 by A.D. Drumm

	        Added preprocessor flag to control whether all directory

	        entries or just files are returned.

*/

 

/*#define FULL_LIST*/

 

#if !defined( _POSIX_SOURCE )

  #define _POSIX_SOURCE

#endif

 

#include <standards.h>

#include <string.h>

#include <dirent.h>

 

/*#define UNIT_TEST*/

 

/* Misc. defines. */

 

#define PATH_MAX                1023

#define PATH_SEPARATOR          '/'

 

/* External functions. */

 

extern  int     scandir( char* name,

	                 struct dirent** list[],

	                 int (*select)(),

	                 int (*compare)() );

extern  int     alphasort( struct dirent** dir1, struct dirent** dir2 );

 

 

/* Local function prototypes. */

 

static  int     match_filename( struct dirent* entry );

 

/* Local static variables. */

 

static  int             current_entry;

static  char*           file;

static  size_t          file_length;

static  char            local_spec[PATH_MAX + 1];

static  int             match_count;

static  struct dirent** name_list;

static  char*           path;

static  int             simple_path;

 

/*--------------------------------------------------------------------------*/

int             findfirst( char* filespec, char* filename )

{

  DIR*          directory;

  char*         pathend;

 

  if( !filespec )

    return -1;

 

  strcpy( local_spec, filespec );

  if( !( pathend = strrchr( local_spec, PATH_SEPARATOR ) ) )

  {

    file = local_spec;

    path = ".";

    simple_path = 1;

  }

  else

  {

    *pathend = '\0';

    file = pathend + 1;

    path = local_spec;

    simple_path = 0;

  }

  file_length = strlen( file );

 

  /* Verify we have a directory name. */

 

  if( !( directory = opendir( path ) ) )

    return -1;

  closedir( directory );

 

  if( !( match_count

	   = scandir( path, &name_list, match_filename, alphasort ) ) )

    return -1;

 

  current_entry = 0;

  if( !simple_path )

  {

    strcpy( filename, path );

    strcat( filename, "/" );

    strcat( filename, name_list[current_entry]->d_name );

  }

  else

    strcpy( filename, name_list[current_entry]->d_name );

  free( name_list[current_entry++] );

  match_count--;

 

  return 0;

}

 

/*--------------------------------------------------------------------------*/

int             findnext( char* filename )

{

  if( !match_count )

    return -1;

 

  if( !simple_path )

  {

    strcpy( filename, path );

    strcat( filename, "/" );

    strcat( filename, name_list[current_entry]->d_name );

  }

  else

    strcpy( filename, name_list[current_entry]->d_name );

  free( name_list[current_entry++] );

  if( !( --match_count ) )

    free( name_list );

  return 0;

}

 

/*--------------------------------------------------------------------------*/

static  int     match_filename( struct dirent* entry )

{

  DIR*          directory;

  char          filename[PATH_MAX + 1];

  char*         scan_entry;

  char*         scan_match;

 

  if( !entry )

    return 0;

 

  if( file_length > entry->d_namlen )

    return 0;

 

  scan_match = file;

  scan_entry = entry->d_name;

  while( *scan_match != '\0' )

  {

    /* If match string has a '*', match all characters until */

    /* the next character in match string is encountered.    */

 

    if( '*' == *scan_match )

    {

      scan_match++;

      while( *scan_entry != *scan_match )

      {

	/* Matching end-of-string is OK but since we didn't match, */

	/* this end-of-string in the entry means we've run out so  */

	/* no match!                                               */

 

	if( '\0' == *scan_entry )

	  return 0;

	scan_entry++;

      }

    }

 

    /* A '?' matches any character and, of course, we continue if */

    /* the corresponding characters do in fact match.             */

 

    else if( '?' == *scan_match

	     || *scan_entry == *scan_match )

    {

      scan_match++;

      scan_entry++;

    }

 

    /* Otherwise, no good so return 0. */

 

    else

      return 0;

  }

 

  /* We matched all of the match string if we got here. */

  /* But we could have some of the entry string left.   */

 

  if( *scan_entry != '\0' )

    return 0;

 

#if defined( FULL_LIST )

  return 1;

#else

  /* Make sure this entry is not itself a directory. */

 

  strcpy( filename, path );

  strcat( filename, "/" );

  strcat( filename, entry->d_name );

  if( !( directory = opendir( filename ) ) )

    return 1;

  closedir( directory );

  return 0;

#endif

}

 

#if defined( UNIT_TEST )

/*--------------------------------------------------------------------------*/

/* This main routine is used to unit test the filelist function.  It is     */

/* included if the UNIT_TEST preprocessor variable is defined.  When run,   */

/* it will prompt for a filespec and will return the entries matching that  */

/* specification.                                                           */

/*--------------------------------------------------------------------------*/

int     main( int argc, char* argv[], char* envp[] )

{

  char          filename[PATH_MAX + 1];

  int           i;

  char          input_line[256];

 

  printf( "Enter filespec (\"done\" to end): " );

  while( gets( input_line ) )

  {

    if( !strcmp( input_line, "done" ) )

      break;

 

    for( i = findfirst( input_line, filename );

	 i != -1;

	 i = findnext( filename ) )

    {

      printf( "[%s] entry: %s\n", input_line, filename );

    }

    printf( "Enter filespec (\"done\" to end): " );

  }

  return 0;

}

#endif

